<?php
/**
 * Copyright since 2007 PrestaShop SA and Contributors
 * PrestaShop is an International Registered Trademark & Property of PrestaShop SA
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License version 3.0
 * that is bundled with this package in the file LICENSE.md.
 * It is also available through the world-wide-web at this URL:
 * https://opensource.org/licenses/AFL-3.0
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * @author    PrestaShop SA and Contributors <contact@prestashop.com>
 * @copyright Since 2007 PrestaShop SA and Contributors
 * @license   https://opensource.org/licenses/AFL-3.0 Academic Free License version 3.0
 */
if (!defined('_PS_VERSION_')) {
    exit;
}

/*
 * Necessary because there's no way to pass the fully namespaced class to WebserviceRequestCore. The class
 * is dynamically created using the first url segment after \api\. When registering resources a class name
 * can be added in the definition but this isn't used if it's a specific management class.
 *
 * Lines WebserviceRequest.php::607-608 in PrestaShop version 1.7.6.5
 * $specificObjectName = 'WebserviceSpecificManagement' . ucfirst(Tools::toCamelCase($this->urlSegment[0]));
 * if (!class_exists($specificObjectName)) { ...
 */

class Aftershipfeed extends Module
{
    /** @var string[] */
    const CONFIG_KEYS = [];

    const ADMIN_CONTROLLERS = [
        [
            'name' => 'Feed for TikTok Shop',
            'visible' => true,
            'class_name' => 'AdminAftershipFeedConfig',
            'parent_class_name' => 'CONFIGURE',
            'icon' => 'trending_up',
        ],
    ];

    /**
     * AfterShip constructor.
     */
    public function __construct()
    {
        $this->module_key = '45dfe3fd6502ad38ab0457689364f6db';
        $this->name = 'aftershipfeed';
        $this->tab = 'market_place';
        $this->version = '1.0.3';
        $this->author = 'AfterShip';
        $this->need_instance = 0;
        $this->ps_versions_compliancy = [
            'min' => '1.6',
            'max' => _PS_VERSION_,
        ];
        $this->bootstrap = true;

        parent::__construct();

        $this->displayName = 'Feed for TikTok Shop';
        $this->description = $this->l('Feed for TikTok Shop module to integrate PrestaShop with AfterShip Feed.');

        $this->confirmUninstall = $this->l('Are you sure you want to uninstall?');

        if (!Configuration::get('AFTERSHIP_FEED')) {
            $this->warning = $this->l('No name provided');
        }
    }

    /**
     * @return bool
     *
     * @throws PrestaShopException
     */
    public function install()
    {
        if (Shop::isFeatureActive()) {
            Shop::setContext(Shop::CONTEXT_ALL);
        }

        // Register hooks first to set up custom webservices so we can set permissions to custom resources.
        return parent::install() &&
            Configuration::updateValue('AFTERSHIP_FEED', 'aftershipfeed') &&
            $this->setupWebservice() &&
            $this->installTabs();
    }

    /**
     * Install all Tabs.
     *
     * @return bool
     */
    public function installTabs()
    {
        foreach (static::ADMIN_CONTROLLERS as $adminTab) {
            if (false === $this->installTab($adminTab)) {
                return false;
            }
        }

        return true;
    }

    /**
     * Install Tab.
     *
     * @param array $tabData
     *
     * @return bool
     */
    public function installTab(array $tabData)
    {
        $tab = new Tab();
        $tab->id_parent = Tab::getIdFromClassName($tabData['parent_class_name']);
        $tab->name = array_fill_keys(array_values(Language::getIDs(false)), $tabData['name']);
        $tab->class_name = $tabData['class_name'];
        // Makes the module accessible in the Admin Controller, see ModuleAdminControllerCore constructor.
        $tab->module = $this->name;
        $tab->active = (bool) $tabData['visible'];
        $tab->icon = $tabData['icon'];

        return $tab->save();
    }

    /**
     * Turn on webservice, create token and set permissions.
     *
     * @return bool
     */
    private function setupWebservice()
    {
        if (
            !Configuration::updateValue('PS_WEBSERVICE', true) ||
            !$this->createWebserviceKey()
        ) {
            return false;
        }

        return true;
    }

    /**
     * Auto-setup webservice key and permissions for API access.
     *
     * @return bool
     */
    private function createWebserviceKey()
    {
        $existingKey = Configuration::get('AFTERSHIP_FEED_WEBSERVICE_KEY');

        // If we've already created a key, just pass.
        if (
            $existingKey &&
            WebserviceKey::keyExists($existingKey)
        ) {
            return true;
        }

        // Create and set the WebserviceKey object properties
        $webservice = new WebserviceKey();
        $key = Tools::passwdGen(32);
        $webservice->key = $key;
        $webservice->description = 'AfterShip Feed webservice key';

        // Save webservice key
        if (
            !$webservice->add() ||
            !Configuration::updateValue('AFTERSHIP_FEED_WEBSERVICE_ID', $webservice->id) ||
            !Configuration::updateValue('AFTERSHIP_FEED_WEBSERVICE_KEY', $webservice->key)
        ) {
            $this->_errors[] =
                $this->l('It was not possible to install the AfterShip Feed module: webservice key creation error.');

            return false;
        }

        // Set webservice key permissions
        if (!$this->setWebservicePermissionsForAccount($webservice->id, $this->getWebservicePermissions())) {
            $this->_errors[] =
                $this->l('It was not possible to install the AfterShip Feed module: webservice key permissions setup error.');

            return false;
        }

        return true;
    }

    /**
     * Get Webservice permissions for AfterShip Feed webservice key.
     *
     * @return string[][]
     */
    private function getWebservicePermissions()
    {
        return [
            'addresses' => ['GET', 'POST'],
            'carriers' => ['GET'],
            'carts' => ['GET', 'POST'],
            'categories' => ['GET'],
            'combinations' => ['GET'],
            'configurations' => ['GET'],
            'countries' => ['GET'],
            'currencies' => ['GET'],
            'customers' => ['GET', 'POST'],
            'languages' => ['GET'],
            'order_carriers' => ['GET', 'PUT'],
            'order_cart_rules' => ['GET'],
            'order_discounts' => ['GET'],
            'order_details' => ['GET', 'PUT'],
            'order_histories' => ['GET', 'DELETE'],
            'order_payments' => ['GET', 'DELETE'],
            'order_states' => ['GET'],
            'orders' => ['GET', 'POST', 'PUT'],
            'product_option_values' => ['GET'],
            'product_options' => ['GET'],
            'products' => ['GET'],
            'shop_urls' => ['GET'],
            'states' => ['GET'],
            'stock_availables' => ['GET'],
            'stores' => ['GET'],
            'suppliers' => ['GET'],
            'tags' => ['GET'],
        ];
    }

    /**
     * Set permissions for resources on auto-created webservice token.
     *
     * @param $webserviceId
     * @param $permissionsToSet
     *
     * @return bool
     *
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     */
    private function setWebservicePermissionsForAccount($webserviceId, $permissionsToSet)
    {
        $success = true;
        $sql = 'DELETE FROM `' . _DB_PREFIX_ . 'webservice_permission` WHERE `id_webservice_account` = ' . (int) $webserviceId;
        if (!Db::getInstance()->execute($sql)) {
            $success = false;
        }
        if (isset($permissionsToSet)) {
            $permissions = [];
            $methods = ['GET', 'PUT', 'POST', 'DELETE', 'HEAD'];
            foreach ($permissionsToSet as $resource_name => $resource_methods) {
                foreach ($resource_methods as $method_name) {
                    if (in_array($method_name, $methods)) {
                        $permissions[] = [$method_name, $resource_name];
                    }
                }
            }
            $account = new WebserviceKey($webserviceId);
            if ($account->deleteAssociations() && $permissions) {
                $sql = 'INSERT INTO `' . _DB_PREFIX_ . 'webservice_permission` (`id_webservice_permission` ,`resource` ,`method` ,`id_webservice_account`) VALUES ';
                foreach ($permissions as $permission) {
                    $sql .= '(NULL , \'' . pSQL($permission[1]) . '\', \'' . pSQL($permission[0]) . '\', ' . (int) $webserviceId . '), ';
                }
                $sql = rtrim($sql, ', ');
                if (!Db::getInstance()->execute($sql)) {
                    $success = false;
                }
            }
        }

        return $success;
    }

    /**
     * @return bool
     */
    public function uninstall()
    {
        if (Shop::isFeatureActive()) {
            Shop::setContext(Shop::CONTEXT_ALL);
        }

        if (
            !parent::uninstall() ||
            !Configuration::deleteByName('AFTERSHIP_FEED') ||
            !$this->deleteAfterShipFeedConfigurationKeys() ||
            !$this->uninstallTabs()
        ) {
            return false;
        }

        return true;
    }

    /**
     * Delete module configuration keys.
     *
     * @return bool
     */
    public function deleteAfterShipFeedConfigurationKeys()
    {
        return count(array_filter(self::CONFIG_KEYS, 'Configuration::deleteByName')) == count(self::CONFIG_KEYS);
    }

    /**
     * Uninstall all Tabs.
     *
     * @return bool
     */
    public function uninstallTabs()
    {
        foreach (static::ADMIN_CONTROLLERS as $adminTab) {
            if (false === $this->uninstallTab($adminTab)) {
                $this->_errors[] = 'Failed to uninstall all tabs.';

                return false;
            }
        }

        return true;
    }

    /**
     * Uninstall Tab.
     *
     * @param array $tabData
     *
     * @return bool
     */
    public function uninstallTab(array $tabData)
    {
        $tabId = Tab::getIdFromClassName($tabData['class_name']);
        $tab = new Tab($tabId);

        if (false === (bool) $tab->delete()) {
            return false;
        }

        return true;
    }

    /**
     * Validate form and handle requests for the displayForm action.
     */
    public function getContent()
    {
        Tools::redirectAdmin($this->context->link->getAdminLink('AdminAftershipFeedConfig'));
    }
}
